import multiprocessing
import os
import queue
from time import sleep, time

import cv2
import numpy as np
from PIL import Image

from attribute_detect.inference import start_attr_det
from config import config as cfg
from face_detect.face_detection_and_recognition_2 import face_detection_and_recognition_2
from face_detect.face_detection_and_recognition_3 import face_detection_and_recognition_3
from utils import LOGGER
from utils.capture import fetch, start_streams
from utils.folder import file_num
from utils.initialize import initialize
from utils.position import is_contain
from yolov3.yolo_detect import set_yolo
from zone import get_zone


def attr_main(cfg):
    zones = get_zone()
    yolo_face_recv = cfg.YOLO.FACE_RECV
    yolo_face_put = cfg.YOLO.FACE_PUT
    yolo_body_recv = cfg.YOLO.BODY_RECV
    yolo_body_put = cfg.YOLO.BODY_PUT

    # 初始化face
    if os.path.exists(cfg.ATTR.ENCODING):
        known_faces = np.load(cfg.ATTR.ENCODING)
    else:
        known_faces = []

    frame_id = 0
    most_recent_ids = []
    most_recent_bodies = []
    while True:
        imgs = fetch(cfg, 'attr', completed=True)
        if len(imgs) != len(cfg.CAMERA.FACE_URLS):
            LOGGER.debug('the number of face image is not equal to the face camera.')
            sleep(1 / cfg.ATTR.CAMERA_FPS)
            continue
        LOGGER.debug('got {:d} frames.'.format(len(imgs)))

        frame_id += 1
        most_recent_ids = [x for x in most_recent_ids if frame_id - x[1] < cfg.ATTR.HIST_NUM]
        most_recent_bodies = [x for x in most_recent_bodies if frame_id - x[1] < cfg.ATTR.HIST_NUM]

        det_face_imgs = [i for i in imgs if i[0] == cfg.ATTR.FACE_CAM]
        yolo_face_recv.put(det_face_imgs, timeout=10)
        try:
            face_results = yolo_face_put.get(timeout=10)
        except queue.Empty:
            LOGGER.warning('no yolo results receive.')
            face_results = []

        for p in face_results:
            cam_id, img_id, x1, y1, x2, y2 = p[:6]
            head_offset = (0.5 * (x1 + x2), y1)
            cam_zones = [z.vertices for z in zones if z.camera_id == cam_id]
            if cam_zones:
                for zone in cam_zones:
                    if is_contain(head_offset, zone):
                        break
                else:
                    LOGGER.info('person [{:d}, {:d}, {:d}, {:d}] at camera-{:s} not in valid zone.'.format(
                        x1, y1, x2, y2, cam_id))
                    continue
            else:
                LOGGER.debug('camera-{:s} not set a valid zone, so it will use the whole image.')

            face_container = det_face_imgs[img_id][1]
            face_img = face_container[y1:y2, x1:x2, ::-1]  # rgb face image
            face_id, known_faces, face_encoding = face_detection_and_recognition_2(face_img, known_faces)
            if face_id is not None:
                most_recent_ids = [x for x in most_recent_ids if x[0] != face_id]
                most_recent_ids.append((face_id, frame_id, face_encoding))

        det_body_imgs = [i for i in imgs if i[0] == cfg.ATTR.BODY_CAM]
        yolo_body_recv.put(det_body_imgs, timeout=10)
        try:
            body_results = yolo_body_put.get(timeout=10)
        except queue.Empty:
            LOGGER.warning('no yolo results receive.')
            body_results = []

        for b in body_results:
            cam_id, img_id, x1, y1, x2, y2 = b[:6]
            body_offset = (0.5 * (x1 + x2), 0.5 * (y1 + y2))
            cam_zones = [z.vertices for z in zones if z.camera_id == cam_id]
            if cam_zones:
                for zone in cam_zones:
                    if is_contain(body_offset, zone):
                        break
                else:
                    LOGGER.debug('person [{:d}, {:d}, {:d}, {:d}] at camera-{:s} not in valid zone.'.format(
                        x1, y1, x2, y2, cam_id))
                    continue
            else:
                LOGGER.debug('camera-{:s} not set a valid zone, so it will use the whole image.')
            body_container = det_body_imgs[img_id][1]
            body_img = body_container[y1:y2, x1:x2, ::-1]  # rgb body image
            most_recent_bodies.append(
                ([cam_id, body_img], frame_id, None))

        most_recent_ids, most_recent_bodies, id_to_body = face_detection_and_recognition_3(most_recent_ids, most_recent_bodies)

        for face_id, bodies in id_to_body.items():
            face_folder = os.path.join(cfg.REID.GALLERY, str(face_id))
            if not os.path.exists(face_folder):
                os.makedirs(face_folder)

            for i, (cam_id, body) in enumerate(bodies[::-1]):
                if file_num(face_folder) < cfg.ATTR.GALLERY_NUM:
                    face_path = os.path.join(face_folder,
                                             '{:d}_{:s}_{:d}.jpg'.format(face_id, cam_id, int(time()) + i))
                    cv2.imwrite(face_path, body[:, :, ::-1])  # 把图片写出入库
                    LOGGER.info('successfully add person image {:d}'.format(face_id))

                    # if (file_num(face_folder) == 1):  # 第一次入库则做属性检测
                    #     pil_body = Image.fromarray(body)
                    #     cfg.ATTR.PIPE.put([pil_body, face_folder, face_id])
                else:
                    LOGGER.info('{:d} already in library.'.format(face_id))


if __name__ == '__main__':
    os.environ['PYTHONWARNINGS'] = 'ignore:semaphore_tracker:UserWarning'
    multiprocessing.set_start_method('forkserver', force=True)
    initialize(cfg)
    start_streams(cfg, ('attr',))
    # start_attr_det(cfg)
    set_yolo(cfg, ('attr',))
    attr_main(cfg)
